from functools import wraps
from flask import request
from flask_jwt_extended import get_jwt_identity, verify_jwt_in_request
from app.models.user import User
from app.models.action_log import ActionLog
from app.utils.response_helper import error_response
import uuid


def admin_required(fn):
    """Decorator to require admin privileges"""
    @wraps(fn)
    def wrapper(*args, **kwargs):
        # Verify JWT token
        verify_jwt_in_request()

        user_id = get_jwt_identity()
        user = User.objects(user_id=user_id, is_active=True).first()

        if not user:
            request_id = str(uuid.uuid4())
            action_log_id = str(uuid.uuid4())

            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user_id,
                action_type='admin_access_denied',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                response_status='failed',
                error_details={'reason': 'User not found'}
            ).save()

            return error_response(
                message="Access denied",
                error_code="FORBIDDEN",
                error_details="Admin access required",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        if user.user_type != 'AD':
            request_id = str(uuid.uuid4())
            action_log_id = str(uuid.uuid4())

            # Log failed action
            ActionLog(
                action_log_id=action_log_id,
                user_id=user_id,
                action_type='admin_access_denied',
                ip_address=request.headers.get('X-Real-IP', request.remote_addr),
                user_agent=request.headers.get('User-Agent', ''),
                response_status='failed',
                error_details={'reason': 'Not an admin user'}
            ).save()

            return error_response(
                message="Access denied",
                error_code="FORBIDDEN",
                error_details="Admin access required",
                status_code=403,
                request_id=request_id,
                action_log_id=action_log_id
            )

        return fn(*args, **kwargs)
    return wrapper
